require_relative "./escape.rb"
require_relative "./type.rb"
require_relative "./version.rb"

class Op < Struct.new(:name, :parse_nargs, :block, :ways)
  def dup
    Op.new(name.dup, parse_nargs, block, ways.map(&:dup))
  end
  def dup_mod(mod)
    return self if mod <= 0
    r = dup
    r.ways.each{|w| w.mod += mod if w.can_mod? }
    r
  end
end

class Way < Struct.new(:name, :arg_ranks, :ret_rank, :arg_types, :ret_type, :mod, :vectorize, :promote, :impl)
  def mod_arg_ranks
    arg_ranks.map{|r| r < 0 ? ~r + mod : r }
  end
  def mod_ret_rank
    ret_rank < 0 ? ~ret_rank + mod : ret_rank
  end
  def dup
    Way.new(name, arg_ranks.dup, ret_rank, arg_types.dup, ret_type, mod, vectorize, promote, impl)
  end
  def can_mod?
    (arg_ranks + [ret_rank]).any?{|r| r < 0 }
  end
end

def lookup_op(name)
  (0..).each{|mod|
    op = Ops[name]
    return nil if op && mod>0 && !op.ways.any?(&:vectorize)
    return op.dup_mod(mod) if op
    return nil if !name[/,$/]
    name = $`
  }
end

def new_op(name, type_sig, block: false, vectorize: true, promote: true, &impl)
  promote = [promote] unless Array === promote
  parts = type_spec_parts(type_sig)
  *arg_ranks, ret_rank = parts.map{|arg| spec_rank(arg) }
  *arg_types, ret_type = parts.map{|arg| spec_type(arg) }

  way = Way.new(name, arg_ranks, ret_rank, arg_types, ret_type, 0, vectorize, promote, impl)
  Op.new(name, arg_ranks.size, block, [way])
end

Ops = {}
OpExamples = {}

if !defined? mk_op
  def mk_op(names, type_sig, example="", block: false, vectorize: true, promote: true, &impl)
    OpExamples[names] = example if example != ""
    names = names.split

    op = new_op(names[-1], type_sig, block: block, vectorize: vectorize, promote: promote, &impl)

    names.each{|name|
      if Ops[name]
        Ops[name].ways << op.ways[0]
      else
        Ops[name] = op.dup
      end
    }

    if op.ways[0].can_mod? && op.ways[0].vectorize
      mod_op = op.dup
      mod_op.ways[0].mod = 1

      names.each{|name|
        if name =~ /^[a-z]/
          name = name.sub(/^./){|first|first.upcase}
          if Ops[name]
            Ops[name].ways << mod_op.ways[0]
          else
            Ops[name] = mod_op.dup
          end
        end
      }
    end
  end

  def category(name); end
  def quickref_op(names, desc, example)
    OpExamples[names] = example if example != ""
  end
end

NotFirst = :not_first
RepSnd = :rep_second

category("mathy ops")
mk_op("+ add", "int x -> x", '1 2+ -> 3', &add=->a,b{ a.value+b.value })
mk_op("+ add", "char int -> char", "'a 1+ -> 'b'", &add)
mk_op("- sub", "x int -> x", '5 3- -> 2', &subt=->a,b{ a.value-b.value })
mk_op("- sub", "char char -> int", "'b 'a- -> 1", &subt)
mk_op("* mult", "int int -> int", '2 3* -> 6'){|a, b| a.value*b.value }
mk_op("/ div", "int int -> int", '7 2/ -> 3'){|a, b| b.value == 0 ? 0 : a.value/b.value }
mk_op("% mod", "x int -> int", '7 2% -> 1'){|a, b| b.value == 0 ? 0 : a.value%b.value }
mk_op("^ pow", "int int -> int", '2 3^ -> 8'){|a, b| pow(a.value, b.value) }
mk_op("_ negate", "int -> int", '1_ -> -1'){|a| -a.value }
mk_op("| abs", "int -> int", '5~| -> 5'){|a| a.value.abs }
mk_op("( pred", "x -> x", '4( -> 3'){|a| a.value-1 }
mk_op(") succ", "x -> x", "'a) -> 'b'"){|a| a.value+1 }
mk_op("} countTo", "int -> [int]", "3} -> [1,2,3]"){|a| range(1,a.value+1) }
mk_op("{ wholes", "[int]", "{ -> [0,1,2,..."){ range_from(0) }
mk_op("# sum", "[x] -> int", '1,2,3,4 # -> 10', promote: false){|a| to_eager_list(a).sum }

category("char ops")
mk_op("^ charRange", "char char -> [char]", "'a'f^ -> \"abcdef\""){|a,b| range(a.value, b.value+1) }
mk_op("_ read", "[char] -> int", '"23"_ -> 23'){|a| read_num(a)[0] }
mk_op("} readAll", "[char] -> [int]", '"1,2 3"} -> [1,2,3]'){|a| split_non_digits(a) }
mk_op("| ord", "char -> int", "'d| -> 100"){|a| a.value }
mk_op("|, chr", "int -> char", "100|, -> 'd'"){|a| a.value }
mk_op("` str", "int -> [char]", '5` -> "5"'){|a| str(a) }
mk_op("` strip", "[char] -> [char]", '" a b\n"` -> "a b"'){|a| strip(a,$at,$rank) }
mk_op("* replicate", "[char] int -> [char]", '"ab"3* -> "ababab"'){|a, b|
  concat(take(b.value,repeat(a).const).const) }
mk_op("* join", "[[char]] [[char]] -> [char]", '1,2,3 " "* -> "1 2 3"', promote: RepSnd){|a, b| join(a,b)}
mk_op("/ luds charClass", "char int -> int", '"Ab$"9/ -> [0,1,1]'){|a,b| char_class(a.value, b.value) }
mk_op("/ split", "[char] [char] -> [[char]]", '"a  b c " " "/ -> ["a","b","c"]'){|a, b| s=cut(a,repeat(b).const).const; filter(s,s,nil,1) }
mk_op("% cut splitKeepEmpties", "[char] [[char]] -> [[char]]",'"a  b c " " "% -> ["a","","b","c",""]', promote: RepSnd){|a, b| cut(a,b) }

category("generic ops")
mk_op("a append", "[a] [a] -> [a]", '"ab" "cd"a -> "abcd"'){|a, b| append(a,b) }
mk_op("b backwards reverse", "[a] -> [a]", '"abc"b -> "cba"', promote: false){|a| reverse(a) }
mk_op("c cons", "[a] a -> [a]", '1,2 0c -> [0,1,2]'){|a, b| [b, a] }
mk_op("d drop", "[a] int -> [a]", '"abcde" 2d -> "cde"'){|a, b| drop(b.value, a) }
mk_op("g get", "[a] int -> a", '"abcd" 2g -> \'c\''){|a, b| z=zero($rank,$at); a = drop(b.value, a); a==[] ? z : a[0].value }
mk_op("h head", "[a] -> a", '"abc"h -> \'a\'', promote: false){|a| z=zero($rank,$at); a.empty ? z : a.value[0].value }
mk_op("j just", "a -> [a]", "1j -> [1]"){|a|[a, [].const]}
mk_op("k keep take", "[a] int -> [a]", '"abcde" 2k -> "ab"'){|a, b| take(b.value, a) }
mk_op("l last", "[a] -> a", '"abc"l -> \'c\'', promote: false){|a| z=zero($rank,$at); last(a, z.const) }
mk_op("n not", "a -> int", '0,1,2n -> [1,0,0]', promote: false){|a| truthy($at, $rank, a.value) ? 0 : 1 }
mk_op("O Or o, or, or2", "int [char] -> [char]", promote: NotFirst){|a,b| truthy($at, 0, a.value) ? str(a) : b.value }
mk_op("o or", "a a -> a", '" bc" \'- o -> "-bc"', promote: NotFirst){|a,b| truthy($at, $rank, a.value) ? a.value : b.value }
mk_op("p pivot transpose", "[[a]] -> [[a]]", '"abc","def"p -> ["ad","be","cf"]', promote: false){|a| transpose(a, $at, $rank) }
mk_op("q quantity len", "[a] -> int", '"abc"q -> 3', promote: false){|a| len(a.value) }
mk_op("r repeat", "a -> [a]", "1r -> [1,1,1,1,1,1..."){|a| repeat(a) }
mk_op("s sortBy", "[a] [b] -> [a]", '"abc","d":len S -> ["d","abc"]', promote: false){|a,b| sort_by(a,b,$rank) }
mk_op("t tail", "[a] -> [a]", '"abc"t -> "bc"', promote: false){|a| a.empty ? [] : a.value[1].value }
mk_op("u unite concat", "[[a]] -> [a]", '"ab","cd","e"u -> "abcde"', promote: false){|a| concat(a) }
mk_op("v vet filter", "[a] [b] -> [a]", '"abcdef":2%v -> "ace"'){|a,b| filter(a,b,$bt,$rank) }
mk_op("w while takeWhile", "[a] [b] -> [a]", '"abc def":w -> "abc"', promote: false){|a,b| take_while(a,b,$bt,$rank) }
mk_op("x exclude setDiff", "[a] [a] -> [a]", '"abra" "ra"x -> "ba"'){|a,b| set_diff(a,b) }
mk_op("y yesNo ifElse", "a b b -> b", '0 1 2 y -> 2', promote: NotFirst){|a,b,c| truthy($at, $rank, a.value) ? b.value : c.value }
mk_op("z zeroPad", "[a] -> [a]", '1,2,3 z -> [1,2,3,0,0,...'){|a| padv(a, zero($rank,$at).const) }
mk_op("= equal", "a a -> int", "1,2,3 1= -> [1,0,0]"){|a,b| spaceship(a, b, $rank) == 0 ? 1 : 0 }
mk_op("< lessThan", "a a -> int", '1,2,3 2< -> [1,0,0]'){|a,b| spaceship(a,b,$rank) == -1 ? 1 : 0 }
mk_op(". index", "[a] a -> int", '"abcd" \'c . -> 3'){|a,b| index(a,b, $rank) }
mk_op("& reshape", "[a] [int] -> [[a]]", '"abcdef" 2& -> ["ab","cd","ef"]', promote: RepSnd){|a,b| reshape(a,b) }
mk_op("? debut isFirst", "[a] -> [int]", '"aardvark"? -> [1,0,1,1,1,0,0,1]', promote: false){|a| isuniq(a) }
mk_op("\\ chunkWhen", "[a] [b] -> [[a]]", '"ab cd e":\\ -> ["ab ","cd ","e"]', promote: false){|a,b| chunk_while(a,b,$bt,$rank) }


category("folding ops") # (return type is type yielded)
mk_op("i iterate", "a -> [a]", '0 i 1,2,3+ -> [0,1,3,6]', block: true)
mk_op("e expand iterate0", "[a]", 'e 1,2,3+ -> [1,3,6]', block: true)
mk_op("f foldr", "a -> [a]", '0 f 1,2,3+ -> 6', block: true)
mk_op("m meld foldr0", "[a]", 'm 1,2,3+ -> 6', block: true)

category("stack ops")
mk_op(": dup", "a -> a", "5: -> 5 5")
mk_op("; mdup", "a -> a", "5;1- -> 4 5", block: true)

category("special symbols")
quickref_op("$", "arg", "5;$+ -> 10 5")
quickref_op("!", "pop from outer stack", "2 5;!+ -> 7 5")
quickref_op(">", "end block", "3;))> -> 5 3")
quickref_op(",", "unvec", '"ab"j, -> ["ab"]')
quickref_op("@", "register get/set", "@( 5@) -> 4 6")
quickref_op("[ ]", "push pop", "5[( ]) -> 4 6")
quickref_op("~", "assignment", "5~a( a) -> 4 6")

# available for char +
category("low rank overloads")
# Available (for int):
# GSV  2 arg
# LZ   1 arg
# FI   block using

mk_op("U uppercase", "char -> char", "'a U -> 'A'"){|a| a=a.value; a>='a'.ord && a<='z'.ord ? a-32 : a }
mk_op("L lowercase", "char -> char", "'A L -> 'a'"){|a| a=a.value; a>='A'.ord && a<='Z'.ord ? a+32 : a }
mk_op("P parts splitWS", "[char] -> [[char]]", '"ab c\n d" P -> ["ab","c","d"]'){|a| x=promise{ group_while(a,a,0,CharType) }; filter(x,x,CharType,1) }

mk_op("p digits", "int -> [int]", '123 p -> [1,2,3]'){|a| v=to_base(a.value,10); v.empty? ? [0.const, Null] : v }
mk_op("U undigits", "[int] -> int", '1,2,3 U -> 123', promote: false){|a| undigits(a) }
mk_op("K kCopy", "int int -> [int]", '4 3K -> [4,4,4]'){|a,b| take(b.value,repeat(a).const) }
mk_op("D digitize toBase", "int int -> [int]", '6 2 D -> [1,1,0]'){|a,b| to_base(a.value,b.value) }
mk_op("W W, baseWas baseFrom", "int [int] -> int", '2 1,1,0 W -> 6', promote: false){|a,b| from_base(b.value,a.value) }
mk_op("B rangeFrom rangeBegin", "x -> [x]", "2B -> [2,3,4,..."){|a| range_from(a.value) }
mk_op("T rangeTo", "int -> [int]", "3T -> [0,1,2]"){|a| range(0,a.value) }
mk_op("u unsquare sqrt", "int -> int", "9u -> 3"){|a| square_root(a.value, 2)  }
mk_op("A min", "x x -> x", '4 5 A -> 4'){|a,b| spaceship(a,b,$rank) <= 0 ? a.value : b.value }
mk_op("X max", "x x -> x", '"bc" "ad" X -> "bd"'){|a,b| spaceship(a,b,$rank) >= 0 ? a.value : b.value }
mk_op("H hyper pow10", "int -> int", "2H -> 100"){|a| pow(10, a.value) }
mk_op("P prod", "[int] -> int", '1,2,3,4 P -> 24', promote: false){|a| to_eager_list(a).inject(1,&:*) }

category("debugging ops")
mk_op("nil", "[a]", 'nil -> []'){ [] }
mk_op("pad", "[a] a -> [a]", '1,2 3 pad -> [1,2,3,3,3...'){|a, b| padv(a, b) }
mk_op("show", "a -> [char]", '1,2 show -> "[1,2]"', vectorize: false, &show=->a{ inspectv($at, $rank, a) })
mk_op("type", "a -> [char]", '1,2 type -> "[int]"', vectorize: false){|a| str_to_lazy_list(type_and_rank_to_str($at, $rank)) }
mk_op("version", "[char]", "version -> \"#{Version}, ruby#{RUBY_VERSION}\""){ str_to_lazy_list(Version + ", ruby#{RUBY_VERSION}") }
mk_op("del", "a -> a", '1 2 del -> 1', vectorize: false){|a| raise}

category("internal ops")
# mk_op("lines", "[char] -> [[char]]"){|a| lines(a) }
mk_op("superpad", "[a] -> [a]", promote: false){ raise }
mk_op("zero", "[] -> int"){|a| 0 } # default to int in fold0s
mk_op("zerom", "[a] -> a"){|a| zero($rank,$at) } # this is used for rank1+ fold0s
mk_op("zero", "[a] -> a"){|a| zero($rank,$at) }

# todo do this without duplicating logic/names
category("intuitive special promotes (don't need to show them in quickref)")
mk_op("* join join2", "[char] [[char]] -> [char]", promote: [RepSnd,NotFirst]){|a, b| join(map(a){|i| [i,Null] }.const,b)} # ideally don't promote first at all
mk_op("S SortBy sortBy, s, sortBy2", "[[a]] [b] -> [[a]]", promote: false){|a,b| sort_by(a,b,$rank) }
mk_op("V Vet Filter vet, filter, v, filter2", "[[a]] [b] -> [[a]]", promote: false){|a,b| filter(a,b,$bt,$rank) }
mk_op("W While TakeWhile w, while, takeWhile, takeWhile2", "[[a]] [b] -> [[a]]", promote: false){|a,b| take_while(a,b,$bt,$rank-1) }
mk_op("\\, ChunkWhen chunkWhen, chunkWhen2", "[[a]] [b] -> [[[a]]]", promote: false){|a,b| chunk_while(a,b,$bt,$rank) }
mk_op("Y y, IfElse ifElse, ifElse2", "[a] b b -> b", promote: NotFirst){|a,b,c| truthy($at, $rank+1, a.value) ? b.value : c.value }

Unlines = Way.new("unlines", [2], [1], [CharType], CharType, 0, nil, nil, ->a{
  concat(map(a){|e| append(e, [10.const, Null].const) }.const)
})
Unwords = Way.new("unwords", [2], [1], [CharType], CharType, 0, nil, nil, ->a{
  join(a, repeat([32.const, Null].const).const)
})